// $Id: CWindow.cpp,v 1.12 2007/03/03 02:59:20 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CWindow.hpp"
using Exponent::GUI::Windowing::CWindow;

#ifdef WIN32
	HWND CWindow::m_hookWindowHandle = NULL;
	#include <Host/CDriveInformation.hpp>
	#include "../Controls/CTextEdit.hpp"
	using Exponent::GUI::Controls::CTextEdit;
	using Exponent::Host::CDriveInformation;
#endif

#include <Host/CApplication.hpp>
using Exponent::Host::CApplication;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION_NO_PARENT(CWindow);

//	===========================================================================
CWindow::CWindow(const CRect &size) 
	   : m_mouseEvent(&m_mouse, CMouseEvent::e_mouseMove)
	   , m_referenceCount(0)
	   , m_theControlRoot(NULL)
	   , m_timedObjects(NULL)
	   , m_toolTip(NULL)
	   , m_windowInitialised(false)
	   , m_windowParent(NULL)
	   , m_windowHandle(NULL)
	   , m_attributes(NULL)
	   , m_focusListener(NULL)
	   , m_windowChangeListener(NULL)
	   , m_keyboardListener(NULL)
	   , m_dropFileListener(NULL)
	   , m_mouseListener(NULL)
#ifdef WIN32
	   , m_setLayeredWindowAttributes(0)
	   , m_keyboardHook(0)	
	   , m_mouseHook(0)		
#else
	   , m_leftClick(false)								
	   , m_timerContexts(NULL)				
	   , m_updateAreaIsValid(true)	
	   , m_userWindow(NULL)		
#endif
{
	EXPONENT_CLASS_CONSTRUCTION(CWindow);

	// NULL all our pointers
	NULL_POINTER(m_mouseListener);
	NULL_POINTER(m_dropFileListener);
	NULL_POINTER(m_keyboardListener);
	NULL_POINTER(m_windowChangeListener);
	NULL_POINTER(m_focusListener);
	NULL_POINTER(m_windowParent);

	// Create our variables
	m_attributes   = new CWindowAttributes();
	m_windowHandle = new SWindowHandle;
	m_windowHandle->m_windowHandle = NULL;

	// Setup the window size
	m_windowDimension.setDimension(size.getWidth(), size.getHeight());
	m_windowPosition.setPoint(size.getLeft(), size.getTop());
	m_windowSize = size;

	// Setup extra variables
	m_windowInitialised = false;
	m_timedObjects		= new TPointerCollection<ITimedObject>(&TPointerCollection<ITimedObject>::TPOINTER_COLLECTION_DEFAULT_NULLER);

	// Setup and intialise the control root
	m_theControlRoot = new CControlRoot(this);

	// Reigster default listeners
	this->registerDropFileListener(m_theControlRoot);
	this->registerKeyboardListener(m_theControlRoot);
	this->registerMouseListener(m_theControlRoot);
	this->registerWindowChangeListener(this);
	this->registerFocusListener(this);

#ifdef WIN32
	CSystemString windowsPath;
	CDriveInformation::getWindowsFolder(windowsPath);
	windowsPath.appendPath("System32\\user32.dll");
	CDLLLoader layeredDLL(windowsPath);
	m_setLayeredWindowAttributes = (SETLAYERATTRIBUTE)layeredDLL.getFunction(CString("SetLayeredWindowAttributes"));
	NULL_POINTER(m_mouseHook);
	NULL_POINTER(m_keyboardHook);
#else
	m_leftClick         = true;		// Default is a left click
	m_timerContexts     = new TPointerCollection<CTimerObject>;
	m_updateAreaIsValid = true;
	m_redrawUpdateArea.setRect(CPoint(0, 0), m_windowDimension);
#endif
}

//	===========================================================================
CWindow::~CWindow()
{
	EXPONENT_CLASS_DESTRUCTION(CWindow);

	// Delete the window etc..
	this->uninitialiseWindow();

	// Null all the listeners and parent
	NULL_POINTER(m_mouseListener);
	NULL_POINTER(m_dropFileListener);
	NULL_POINTER(m_keyboardListener);
	NULL_POINTER(m_windowChangeListener);
	NULL_POINTER(m_focusListener);
	NULL_POINTER(m_windowParent);

	// Free variables we created
	FREE_POINTER(m_attributes);
	FREE_POINTER(m_windowHandle);
	FREE_POINTER(m_timedObjects);
	FREE_POINTER(m_theControlRoot);

#ifdef WIN32
	// And on windows we have the set layered window function pointer...
	NULL_POINTER(m_setLayeredWindowAttributes);
#else
	FREE_POINTER(m_timerContexts);
#endif
}

//	===========================================================================
void CWindow::registerChildWindow(IWindow *child)
{
#ifdef WIN32

#else

#endif
}

//	===========================================================================
void CWindow::unregisterAllChildWindows()
{
#ifdef WIN32

#else

#endif
}

//	===========================================================================
void CWindow::unregisterChldWindow(IWindow *child)
{
#ifdef WIN32

#else

#endif
}

//	===========================================================================
void CWindow::initialiseWindow(CWindowAttributes *windowAttributes, const bool showWindow)
{
	try
	{
		if (windowAttributes && !m_windowInitialised)
		{
			*m_attributes = *windowAttributes;
			if (this->createWindow() && showWindow)
			{
				this->openWindow();
				this->updateWindow();
			}
		}
	}
	catch(CException theException)
	{
		CString message;
		message = "Caught an exception whilst initialising window. Further details below:\n\n";
		message.appendString(theException.getErrorReason());

		CString function;
		function = "void CWindow::intialiseWindow(CWindowAttributes *, const bool) ->\n";
		function.appendString(theException.getFunctionName());

		throw CException(message, function);
	}
}

//	===========================================================================
void CWindow::uninitialiseWindow()
{
	if (m_windowInitialised)
	{
		this->unregisterAllChildWindows();
		this->clearTimedObjects();
		this->destroyWindow();
		m_windowInitialised = false;
	}
}

//	===========================================================================
void CWindow::openWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		ShowWindow(m_windowHandle->m_windowHandle, SW_SHOW);
	#else
		ShowWindow(m_windowHandle->m_windowHandle);
		this->enableWindow();
		SelectWindow(m_windowHandle->m_windowHandle);
	#endif
	}
}

//	===========================================================================
void CWindow::closeWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		ShowWindow(m_windowHandle->m_windowHandle, SW_HIDE);
	#else
		HideWindow(m_windowHandle->m_windowHandle);
	#endif
		// Notify the listener
		if (m_windowChangeListener)
		{
			m_windowChangeListener->windowClosed(this);
		}
	}
}

//	===========================================================================
void CWindow::updateWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		UpdateWindow(m_windowHandle->m_windowHandle);
	#else
		// On mac we have to force an update event
		this->redrawWindow();
	#endif
	}
}

//	===========================================================================
void CWindow::disableWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		EnableWindow(m_windowHandle->m_windowHandle, FALSE);
	#else
		ActivateWindow(m_windowHandle->m_windowHandle, FALSE);
	#endif
	}
}

//	===========================================================================
void CWindow::enableWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		EnableWindow(m_windowHandle->m_windowHandle, TRUE);
	#else
		ActivateWindow(m_windowHandle->m_windowHandle, TRUE);
	#endif
	}
}

//	===========================================================================
void CWindow::gainedFocus(IWindow *window)
{
	// Overloaders can do what they fancy..
}

//	===========================================================================
void CWindow::lostFocus(IWindow *window)
{
	// Overloaders can do what they fancy..
}

//	===========================================================================
void CWindow::draw(CGraphics &graphics)
{
	m_theControlRoot->drawRootControl(graphics);
}

//	===========================================================================
void CWindow::redrawWindow(const CRect &area)
{
	if (m_windowInitialised)
	{
		CRect r = area;
	#ifdef WIN32
		RECT rr = r.getAsRect();
		InvalidateRect(m_windowHandle->m_windowHandle, &rr, FALSE);
	#else
		m_updateAreaIsValid = true;
		m_redrawUpdateArea.unionWith(area);
		if (m_userWindow)
		{
			HIRect rect;
			rect.origin.x    = (float)r.getLeft();
			rect.origin.y    = (float)r.getTop();
			rect.size.width  = (float)r.getWidth();
			rect.size.height = (float)r.getHeight();
			HIViewSetNeedsDisplayInRect (m_userWindow, &rect, true);		
		}
		else
		{
			Rect *rr = r.getAsRect();
			InvalWindowRect (m_windowHandle->m_windowHandle,  rr);
			FREE_POINTER(rr);
		}
	#endif
	}
}

//	===========================================================================
void CWindow::redrawWindow()
{
	if (m_windowInitialised)
	{
	#ifdef WIN32
		RECT rr = m_windowSize.getAsRect();
		InvalidateRect(m_windowHandle->m_windowHandle, &rr, FALSE);
	#else
		m_updateAreaIsValid = true;
		m_redrawUpdateArea.setRect(CPoint(0, 0), m_windowDimension);
		if (m_userWindow)
		{
			HIRect rect;
			rect.origin.x    = 0.f;
			rect.origin.y    = 0.f;
			rect.size.width  = (float)m_windowSize.getWidth();
			rect.size.height = (float)m_windowSize.getHeight();
			HIViewSetNeedsDisplayInRect (m_userWindow, &rect, true);
		}
		else
		{
			Rect *rr = m_windowSize.getAsRect();
			InvalWindowRect (m_windowHandle->m_windowHandle,  rr);
			FREE_POINTER(rr);
		}
	#endif
	}
}

//	===========================================================================
void CWindow::setWindowPosition(const CPoint &position)
{
	if (m_windowInitialised)
	{
		m_windowPosition = position;
		m_windowSize.setOrigin(position);
	#ifdef WIN32
		SetWindowPos(m_windowHandle->m_windowHandle, NULL, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), 0, 0, SWP_NOOWNERZORDER | SWP_NOZORDER | SWP_NOSIZE);
	#else
		MoveWindow(m_windowHandle->m_windowHandle, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), false);
	#endif
		// Notify the listener
		if (m_windowChangeListener)
		{
			m_windowChangeListener->windowMoved(this, position);
		}
	}
}

//	===========================================================================
void CWindow::setWindowSize(const CDimension &dimension)
{
	if (m_windowInitialised)
	{
		m_windowDimension = dimension;
		m_windowSize.setWidth(dimension.getWidth());
		m_windowSize.setHeight(dimension.getHeight());
	#ifdef WIN32
		MoveWindow(m_windowHandle->m_windowHandle, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), m_windowDimension.getWidth(), m_windowDimension.getHeight(), TRUE);
	#else
		SizeWindow(m_windowHandle->m_windowHandle, m_windowDimension.getWidth(), m_windowDimension.getHeight(), true);
	#endif
		// Notify the listener
		if (m_windowChangeListener)
		{
			m_windowChangeListener->windowSized(this, dimension);
		}
	}
}

//	===========================================================================
void CWindow::setWindowPositionAndSize(const CRect &positionAndSize)
{
	if (m_windowInitialised)
	{
		m_windowSize = positionAndSize;
		m_windowPosition.setPoint(positionAndSize.getLeft(), positionAndSize.getTop());
		m_windowDimension.setDimension(positionAndSize.getWidth(), positionAndSize.getHeight());
	#ifdef WIN32
		MoveWindow(m_windowHandle->m_windowHandle, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), m_windowDimension.getWidth(), m_windowDimension.getHeight(), TRUE);
	#else
		MoveWindow(m_windowHandle->m_windowHandle, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), false);
		SizeWindow(m_windowHandle->m_windowHandle, m_windowDimension.getWidth(), m_windowDimension.getHeight(), true);
	#endif
		// Notify the listener
		if (m_windowChangeListener)
		{
			m_windowChangeListener->windowMoved(this, m_windowPosition);
			m_windowChangeListener->windowSized(this, m_windowDimension);
		}
	}
}

//	===========================================================================
void CWindow::setWindowAlpha(const double alpha)
{
	if (m_windowInitialised)
	{
		m_attributes->setWindowAlphaValue(alpha);
	#ifdef WIN32
		if (m_setLayeredWindowAttributes != NULL)
		{
			const BYTE theAlpha = (BYTE)(alpha * 255.0);
			m_setLayeredWindowAttributes(m_windowHandle->m_windowHandle, 0, theAlpha, LWA_ALPHA);
		}
	#else
		SetWindowAlpha(m_windowHandle->m_windowHandle, (float)alpha);
	#endif
	}
}

//	===========================================================================
void CWindow::handleTimerEvent(const long id)
{
	const long numberOfTimedObjects = m_timedObjects->getArraySize();
	for (long i = 0; i < numberOfTimedObjects; i++)
	{
		ITimedObject *timed = (ITimedObject *)m_timedObjects->elementAtIndex(i);
		if (timed && timed->getTimerId() == id)
		{
			timed->timerExpired();
		}
	}
}

//	===========================================================================
void CWindow::startTimer(const long id, const long timeInMilliseconds)
{
#ifdef WIN32
	if (id == 0)
	{
		throw CException("Timer id's cannot be zero on Windows platforms", "CWindow::startTimer(const long, const long)");
	}
	SetTimer(m_windowHandle->m_windowHandle, id, timeInMilliseconds, (TIMERPROC)CWindow::TimerProc);
#else
	ITimedObject *theObject = NULL;
	long index = 0;
	for (long i = 0; i < m_timedObjects->getArraySize(); i++)
	{
		ITimedObject *timed = (ITimedObject *)m_timedObjects->elementAtIndex(i);
		if (timed && timed->getTimerId() == id)
		{
			theObject = timed;
			index     = i;
			break;
		}
	}

	if (theObject == NULL)
	{
		return;
	}

	CFRunLoopTimerContext timerContext;
	timerContext.version		 = 0;
	timerContext.info			 = theObject;
	timerContext.retain			 = NULL;
	timerContext.release		 = NULL;
	timerContext.copyDescription = NULL;

	CTimerObject *object = m_timerContexts->elementAtIndex(index);
	if (object)
	{
		object->m_timer    = CFRunLoopTimerCreate(kCFAllocatorDefault, 0.0, timeInMilliseconds / 1000.0, 0, 0, handleTimerMessages, &timerContext);
		object->m_isActive = true;
		CFRunLoopAddTimer((CFRunLoopRef)GetCFRunLoopFromEventLoop(GetMainEventLoop()), object->m_timer, kCFRunLoopCommonModes);
	}
#endif
}

//	===========================================================================
void CWindow::stopTimer(const long id)
{
	if (m_timedObjects == NULL)
	{
		return;
	}
	const long numberOfTimedObjects = m_timedObjects->getArraySize();
	for (long i = 0; i < numberOfTimedObjects; i++)
	{
		ITimedObject *timed = (ITimedObject *)m_timedObjects->elementAtIndex(i);
		if (timed && timed->getTimerId() == id)
		{
		#ifdef WIN32
			KillTimer(m_windowHandle->m_windowHandle, id);
		#else
			CTimerObject *object = m_timerContexts->elementAtIndex(i);
			if (object && object->m_isActive)
			{
				CFRunLoopRemoveTimer((CFRunLoopRef)GetCFRunLoopFromEventLoop(GetMainEventLoop()), object->m_timer, kCFRunLoopCommonModes);
				m_timerContexts->deletePointerAtIndex(i);
			}
		#endif
		}
	}
}

//	===========================================================================
#ifndef WIN32
void CWindow::handleTimerMessages(CFRunLoopTimerRef timer, void *info)
{
	if (info)
	{
		((ITimedObject *)info)->timerExpired();
	}
}
#endif

//	===========================================================================
void CWindow::addTimedObject(ITimedObject *timedObject)
{
	m_timedObjects->addElement(timedObject);
#ifndef WIN32
	m_timerContexts->addElement(new CTimerObject());
#endif
}

//	===========================================================================
void CWindow::removeTimedObject(ITimedObject *timedObject)
{
	if (m_timedObjects == NULL)
	{
		return;
	}
	this->stopTimer(timedObject->getTimerId());

	const long index = m_timedObjects->getIndexOfPointer(timedObject);
	m_timedObjects->deletePointerAtIndex(index);
	m_timedObjects->reorder();

#ifndef WIN32
	m_timerContexts->deletePointerAtIndex(index);
	m_timerContexts->reorder();
#endif
}

//	===========================================================================
void CWindow::clearTimedObjects()
{
	const long numberOfTimedObjects = m_timedObjects->getArraySize();
	for (long i = 0; i < numberOfTimedObjects; i++)
	{
		ITimedObject *timed = (ITimedObject *)m_timedObjects->elementAtIndex(i);
		if (timed)
		{
		#ifdef WIN32
			KillTimer(m_windowHandle->m_windowHandle, timed->getTimerId());
		#else
			CTimerObject *object = m_timerContexts->elementAtIndex(i);
			if (object)
			{
				if (object->m_isActive)
				{
					CFRunLoopRemoveTimer(CFRunLoopGetCurrent(), object->m_timer, kCFRunLoopCommonModes);
				}
				m_timerContexts->deletePointerAtIndex(i);
			}
		#endif
		}
	}
	m_timedObjects->clearArray();
	#ifndef WIN32
		m_timerContexts->clearArray();
	#endif
}

//	===========================================================================
void CWindow::windowSized(IWindow *window, const CDimension &newDimension)
{
	// Nothing to do...
}

//	===========================================================================
void CWindow::windowMoved(IWindow *window, const CPoint &newTopLeft)
{
	// Nothing to do...
}

//	===========================================================================
bool CWindow::windowClosed(IWindow *window)
{
#ifdef WIN32
	if (((IWindow *)this) == CApplication::getApplicationWindow())
	{
		CApplication::stopApplicationEventLoop();
	}
#else
	// On the mac they are tied to a menuing system, windows can be closed and it not end the run time...
#endif
	return true;
}

//	===========================================================================
void CWindow::windowOpened(IWindow *window)
{
	// Nothing to do
}

//	===========================================================================
void CWindow::windowMinimised(IWindow *window)
{
	// Nothing to do
}

//	===========================================================================
void CWindow::windowMaximised(IWindow *window)
{
	// Nothing to do
}

//	===========================================================================
bool CWindow::createWindow()
{
#ifdef WIN32
	WNDCLASSEX windowClass;
	CString myWindowName = "com.exponent.";
	myWindowName.appendString(CApplication::getApplicationName());//m_attributes->getApplicationInstance()->getApplicationName());
	myWindowName.appendString(".");
	myWindowName.appendString(m_attributes->getWindowTitle());

	// Check if it is already registered
	// m_attributes->getApplicationInstance()->getApplicationInstance()
	if (!GetClassInfoEx(CApplication::getApplicationInstance(), _T(myWindowName.getString()), &windowClass))
	{
		memset(&windowClass, 0, sizeof(WNDCLASSEX));

		windowClass.style = CS_DBLCLKS;
		if (m_attributes->windowHasShadow())
		{
			windowClass.style |= CS_DROPSHADOW;
		}

		windowClass.cbSize		  = sizeof(WNDCLASSEX);
		windowClass.lpfnWndProc   = WindowProc;
		windowClass.hIcon         = (m_attributes->getWindowIcon()) ? m_attributes->getWindowIcon()->getIcon() : NULL;
		windowClass.hIconSm		  = (m_attributes->getWindowIcon()) ? m_attributes->getWindowIcon()->getIcon() : NULL;
		windowClass.hCursor		  = LoadCursor(NULL, IDC_ARROW);
		windowClass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);			// Make null and draw in erase bkground
		windowClass.lpszClassName = _T(myWindowName.getString());
		windowClass.hInstance	  = CApplication::getApplicationInstance();//m_attributes->getApplicationInstance()->getApplicationInstance();

		// And register
		if (!RegisterClassEx(&windowClass))
		{
			return false;
		}
	}

	// Create the mdi structure
	MDICREATESTRUCT mdiCreateStruct;
	memset(&mdiCreateStruct, 0, sizeof(mdiCreateStruct));
	mdiCreateStruct.lParam = (LPARAM)this;

	HWND parent = NULL;
	if (m_windowParent)
	{
		parent = m_windowParent->getWindowHandle()->m_windowHandle;
	}

	// Create the window
	m_windowHandle->m_windowHandle = CreateWindowEx(m_attributes->getWindowStyleEx(),
													_T(myWindowName.getString()),
													_T(m_attributes->getWindowTitle().getString()),
													m_attributes->getWindowStyle(),
													m_windowPosition.getXPosition(),
													m_windowPosition.getYPosition(),
													m_windowDimension.getWidth(),
													m_windowDimension.getHeight(),
													parent,
													NULL,
													CApplication::getApplicationInstance(),//m_attributes->getApplicationInstance()->getApplicationInstance(),
													&mdiCreateStruct);

	// Check that the window is valid, if not throw and exception (cos we're screwed if it happens now ;)
	if (m_windowHandle->m_windowHandle == NULL)
	{
		char buffer[256];
		FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, GetLastError(), 0, buffer, 256, NULL);
		throw CException(buffer, "CWindow::createWindow()");
		return false;
	}

	// Install the hooks in there....
	m_keyboardHook	   = SetWindowsHookEx(WH_KEYBOARD, KeyboardProc,   CApplication::getApplicationInstance(), 0);
	m_mouseHook		   = SetWindowsHookEx(WH_MOUSE,    MouseWheelProc, CApplication::getApplicationInstance(), 0);
	m_hookWindowHandle = m_windowHandle->m_windowHandle;

	// Window is initialised
	m_windowInitialised = true;

	// Set the initial alpha value
	this->setWindowAlpha(m_attributes->getWindowAlphaValue());

	// Is the window enabled?
	if (!m_attributes->isWindowEnabled())
	{
		this->disableWindow();
	}

	// Set hte background colour in the graphics
	m_graphics.setBackgroundColour(m_attributes->getBackgroundColour());
	return true;
#else
	const static EventTypeSpec eventSpec[] =
	{
		{ kEventClassMouse, kEventMouseDown					},
		{ kEventClassMouse, kEventMouseUp					},
		{ kEventClassMouse, kEventMouseMoved				},
		{ kEventClassMouse, kEventMouseDragged				},
		{ kEventClassMouse, kEventMouseWheelMoved			},

		{ kEventClassKeyboard, kEventRawKeyDown				},
		{ kEventClassKeyboard, kEventRawKeyUp				},
		{ kEventClassKeyboard, kEventRawKeyRepeat			},

		{ kEventClassWindow, kEventWindowDrawFrame			},
		{ kEventClassWindow, kEventWindowDrawContent		},
		{ kEventClassWindow, kEventWindowClosed				},
		{ kEventClassWindow, kEventWindowShown				},
		{ kEventClassWindow, kEventWindowDeactivated		},
		{ kEventClassWindow, kEventWindowFocusAcquired		},
		{ kEventClassWindow, kEventWindowFocusRelinquish	},
		{ kEventClassWindow, kEventWindowActivated			},
		{ kEventClassWindow, kEventWindowInit				}
	};

	Rect contentRect;
	SetRect(&contentRect, m_windowPosition.getXPosition(), m_windowPosition.getYPosition(), m_windowSize.getRight(), m_windowSize.getBottom());
	if (CreateNewWindow(m_attributes->getWindowClass(), m_attributes->getWindowStyle(), &contentRect, &(m_windowHandle->m_windowHandle)) == noErr)
	{
		// Install the event handlers
		InstallWindowEventHandler(m_windowHandle->m_windowHandle, NewEventHandlerUPP(WindowProc), GetEventTypeCount(eventSpec), eventSpec, this, NULL);
		InstallReceiveHandler(NewDragReceiveHandlerUPP(handleDropMessages), m_windowHandle->m_windowHandle, this);
		//InstallTrackingHandler(NewDragTrackingHandlerUPP(handleDragTrackingMessages), m_windowHandle->m_windowHandle, this);

		// Set the window title
		CString str				= m_attributes->getWindowTitle();
		CFStringRef titleKey    = str.getStringAsCFStringRef();
		CFStringRef windowTitle = CFCopyLocalizedString(titleKey, NULL);
		SetWindowTitleWithCFString(m_windowHandle->m_windowHandle, windowTitle);
		CFRelease(titleKey);
		CFRelease(windowTitle);

		// initialised and we are done..
		m_windowInitialised = true;

		// Is the window enabled?
		if (!m_attributes->isWindowEnabled())
		{
			this->disableWindow();
		}

		// Set the initial alpha value
		this->setWindowAlpha(m_attributes->getWindowAlphaValue());

		// We are done :)
		return true;
	}

	throw CException("Failed to create window structure", "CWindow::createWindow()");
	return false;
#endif
}

//	===========================================================================
void CWindow::destroyWindow()
{
#ifdef WIN32
	// Remove the hooks
	if (m_keyboardHook)
	{
		UnhookWindowsHookEx(m_keyboardHook);
	}
	if (m_mouseHook)
	{
		UnhookWindowsHookEx(m_mouseHook);
	}
	// Destroy the actual window
	DestroyWindow(m_windowHandle->m_windowHandle);
#else
	ReleaseWindow(m_windowHandle->m_windowHandle);
#endif
	NULL_POINTER(m_windowHandle->m_windowHandle);
}

#ifdef WIN32
//	===========================================================================
LRESULT CALLBACK CWindow::WindowProc(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	// This is the window that we are going to pass events to
	CWindow *window = NULL;

	if (message == WM_NCCREATE)
	{
		MDICREATESTRUCT *mdic = (MDICREATESTRUCT *)((LPCREATESTRUCT)lParam)->lpCreateParams;
		window = (CWindow *)(mdic->lParam);
		SetWindowLongPtr(windowHandle, GWL_USERDATA, (LONG)window);
	}
	else
	{
		window = (CWindow *)GetWindowLongPtr(windowHandle, GWL_USERDATA);
	}

	// Do we have a valid window?
	if (window)
	{
		return window->handleWindowEvents(windowHandle, message, wParam, lParam);
	}
	return DefWindowProc(windowHandle, message, wParam, lParam);
}

//	===========================================================================
LRESULT CALLBACK CWindow::KeyboardProc(int nCode, WPARAM wParam, LPARAM lParam)
{
	CWindow *window = (CWindow *)GetWindowLongPtr(CWindow::m_hookWindowHandle, GWL_USERDATA);

	if (window)
	{
		// then we should process keyboard events
		return window->handleKeyboardEvents(nCode, wParam, lParam);
	}
	return 0;
}

//	===========================================================================
LRESULT CALLBACK CWindow::MouseWheelProc(int nCode, WPARAM wParam, LPARAM lParam)
{
	CWindow *window = (CWindow *)GetWindowLongPtr(CWindow::m_hookWindowHandle, GWL_USERDATA);
	if (window)
	{
		// then we should process keyboard events
		return window->handleMouseWheelEvents(nCode, wParam, lParam);
	}
	return 0;
}

//	===========================================================================
VOID CALLBACK CWindow::TimerProc(HWND windowHandle, UINT message, UINT_PTR eventID, DWORD dwTime)
{
	//CTrace::trace("Timer proc");
	CWindow *window = (CWindow *)GetWindowLongPtr(windowHandle, GWL_USERDATA);
	if (window)
	{
		window->handleTimerEvent((long)eventID);
	}
}

//	===========================================================================
LRESULT CWindow::handleKeyboardEvents(int nCode, WPARAM wParam, LPARAM lParam)
{
	// If less than 0 we have to pass down the chain
	if (nCode < 0 || m_keyboardListener == NULL)
	{
		return CallNextHookEx(m_keyboardHook, nCode, wParam, lParam);
	}

	// Get the current key state
	BYTE keyboardState[256];
    GetKeyboardState(keyboardState);

	// Convert it to ascii
	WORD myCharacter;
	ToAscii((UINT)wParam, (UINT)lParam, keyboardState, &myCharacter, 0);

	char character = (char)myCharacter;
	m_keyEvent.setKey(character);
	m_keyEvent.setModifiers(GetAsyncKeyState(17) ? true : false, GetAsyncKeyState(16) ? true : false, GetAsyncKeyState(18) ? true : false);

	bool handled = false;

	// Mouse up or mouse down?
	if (HIWORD(lParam) & KF_UP)
	{
		handled = m_keyboardListener->handleKeyUp(m_keyEvent);
	}
	else
	{
		handled = m_keyboardListener->handleKeyDown(m_keyEvent);
	}

	// Did we handle the event??
	if (!handled)
	{
		return CallNextHookEx(m_keyboardHook, nCode, wParam, lParam);
	}
	else
	{
		return 1;	// Processed sucessfully...
	}
}

//	===========================================================================
LRESULT CWindow::handleMouseWheelEvents(int nCode, WPARAM wParam, LPARAM lParam)
{
	// If less than 0 we have to pass down the chain
	if (nCode < 0 || m_keyboardListener == NULL)
	{
		return CallNextHookEx(m_mouseHook, nCode, wParam, lParam);
	}

	// Did we handle the events
	bool handled = false;

	// Is it a mouse wheel?
	if (wParam == WM_MOUSEWHEEL)
	{
		// Do we have a listener?
		if (m_mouseListener)
		{
			// Get the position ofthe mouse...
			POINT point;
			GetCursorPos(&point);

			// Get window left and right
			CRect area;
			CWindowTools::getGlobalContentAreaCoordinates(m_windowHandle, area);

			m_mousePoint.setPoint(point.x - area.getLeft(), point.y - area.getTop());
			m_mouse.setPosition(m_mousePoint);
			m_mouseEvent.setModifiers(GetAsyncKeyState(17) ? true : false, GetAsyncKeyState(16) ? true : false, GetAsyncKeyState(18) ? true : false);

			// Get the hook events
			MOUSEHOOKSTRUCTEX *myHookEvents = (MOUSEHOOKSTRUCTEX*)lParam;

			// Convert and set the wheel amount
			double delta = (double)GET_WHEEL_DELTA_WPARAM(myHookEvents->mouseData);
			const static double wheelDelta = 1.0 / (double)WHEEL_DELTA;
			m_mouseEvent.setWheelMovementAmount(delta * wheelDelta);

			// Pass to the listener for handling
			m_mouseListener->handleMouseScroll(m_mouseEvent);

			// we are done!
			handled = true;
		}
	}

	// Did we handle the event??
	if (!handled)
	{
		return CallNextHookEx(m_mouseHook, nCode, wParam, lParam);
	}
	else
	{
		return 1;	// Processed sucessfully...
	}
}

//	===========================================================================
LRESULT CWindow::handleWindowEvents(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	// Keyboard handling needs to have a hook as does mouse wheel...
	LRESULT result = -1;
	switch(message)
	{
		case WM_CTLCOLOREDIT:
		case WM_CTLCOLORSTATIC:
			{
				IControl *control = this->getControlRoot()->getLockedControl();
				if (control->getObjectClass().isTypeOf("CTextEdit"))
				{
					return ((CTextEdit *)control)->setEditWindowDrawConditions((HDC)wParam);
				}
			}
		break;
		case WM_PAINT:
			{
				// Get our initial DC
				PAINTSTRUCT paintStructure;
				HDC drawContext = BeginPaint(m_windowHandle->m_windowHandle, &paintStructure);

				// Get the update area
				CRect updateArea;
				updateArea.setFromRect(paintStructure.rcPaint);
				m_graphics.setUpdateArea(updateArea);

				// Initialise the double buffer contained within m_graphics (see CNativeImage)
				m_graphics.initialise(m_windowHandle, drawContext, m_windowDimension);

				// Reset the graphics drawing area
				m_graphics.resetDrawingArea();

				// Clip to the update area
				m_graphics.getClippingRegion().setCurrentClipRegion(updateArea);

				// Check if we actuall have to draw anything
				if (updateArea.isRectangleEmpty())
				{
					// Overloaded draw...
					draw(m_graphics);

					// Blit it back to the main window
					BitBlt(drawContext, 0, 0, updateArea.getWidth(), updateArea.getHeight(), m_graphics.getMutableNativeImage()->getGraphicsHandle()->m_drawContext, 0, 0, SRCCOPY);
				}
				else
				{
					// Overloaded draw...
					draw(m_graphics);

					// Blit it back to the window
					BitBlt(drawContext, updateArea.getLeft(), updateArea.getTop(), updateArea.getWidth(), updateArea.getHeight(), m_graphics.getMutableNativeImage()->getGraphicsHandle()->m_drawContext, updateArea.getLeft(), updateArea.getTop(), SRCCOPY);
				}

				// Reset the graphics drawing area
				m_graphics.resetDrawingArea();

				// Clear the old clipping
				m_graphics.getClippingRegion().clearClipRegion();

				// Delete the DC's
				m_graphics.uninitialise();

				// And we are done..
				EndPaint(m_windowHandle->m_windowHandle, &paintStructure);

				// ...completey!
				result = 0;
			}
		break;
		case WM_ERASEBKGND:
			{
				HDC drawContext = (HDC)wParam;
				RECT area		= { 0, 0, m_windowDimension.getWidth(), m_windowDimension.getHeight() };
				HBRUSH brush    = CreateSolidBrush(m_attributes->getBackgroundColour().getAsColourRef());
				HGDIOBJ old		= SelectObject(drawContext, brush);
				FillRect(drawContext, &area, brush);
				SelectObject(drawContext, old);
				DeleteObject(brush);
				result = 1;
			}
		break;
		case WM_DROPFILES:
			if (m_dropFileListener)
			{
				// Get the files and the number of them
				HDROP dropFiles = (HDROP)wParam;
				long numberOfFiles = (long)DragQueryFile(dropFiles, 0xFFFFFFFF, NULL, 0);

				// Get the position of the drop
				CPoint position;
				POINT winPoint;
				DragQueryPoint(dropFiles, &winPoint);
				position.setPoint(winPoint.x, winPoint.y);

				// Stuff them in to an array
				CDropEvent dropEvent(position);
				TStringCountedPointerArray *array = dropEvent.getDroppedFileNames();
				for (long i = 0; i < numberOfFiles; i++)
				{
					char text[2048];
					memset(text, 0, 2048 * sizeof(char));
					DragQueryFile(dropFiles, i, text, 2048);
					array->addElement(new CString(text));
				}

				// Now let the listener handle
				m_dropFileListener->handleFileDrop(dropEvent);

				// Notify that we are done..
				DragFinish(dropFiles);
				SetForegroundWindow(m_windowHandle->m_windowHandle);

				result = 0;
			}
		break;
		case WM_SETFOCUS:
			if (m_focusListener)
			{
				m_focusListener->gainedFocus(this);
			}
			/*
			if (m_keyboardHook)
			{
				this->unhookKeyboardHandler();
			}
			if (m_mouseHook)
			{
				this->unhookWheelHandler();
			}*/
			//m_keyboardHook	   = SetWindowsHookEx(WH_KEYBOARD, KeyboardProc,   CApplication::getApplicationInstance(), 0);
			//m_mouseHook		   = SetWindowsHookEx(WH_MOUSE,    MouseWheelProc, CApplication::getApplicationInstance(), 0);
			//m_hookWindowHandle = windowHandle;
			result = DefWindowProc(windowHandle, message, wParam, lParam);
		break;
		case WM_KILLFOCUS:
			//this->unhookKeyboardHandler();
			//this->unhookWheelHandler();
			if (m_focusListener)
			{
				m_focusListener->lostFocus(this);
			}
			result = DefWindowProc(windowHandle, message, wParam, lParam);
		break;
		case WM_DESTROY:
			if (m_windowChangeListener)
			{
				if (m_windowChangeListener->windowClosed(this))
				{
					this->unhookKeyboardHandler();
					this->unhookWheelHandler();
					result = DefWindowProc(windowHandle, message, wParam, lParam);
				}
			}
			else
			{
				this->unhookKeyboardHandler();
				this->unhookWheelHandler();
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_CLOSE:
			//this->unhookKeyboardHandler();
			//this->unhookWheelHandler();
			if (m_windowChangeListener)
			{
				if (m_windowChangeListener->windowClosed(this))
				{
					result = DefWindowProc(windowHandle, message, wParam, lParam);
				}
			}
			else
			{
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_SIZE:
			{
				m_windowDimension = CDimension(LOWORD(lParam), HIWORD(lParam));
				m_windowSize.setWidth(m_windowDimension.getWidth());
				m_windowSize.setHeight(m_windowDimension.getHeight());
				if (m_windowChangeListener)
				{
					m_windowChangeListener->windowSized(this, m_windowDimension);
				}
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_MOVE:
			{
				const CPoint point(LOWORD(lParam), HIWORD(lParam));
				m_windowPosition = point;
				m_windowSize.setOrigin(m_windowPosition);

				if (m_windowChangeListener)
				{
					m_windowChangeListener->windowMoved(this, point);
				}
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_MOUSEMOVE:
			if (m_mouseListener)
			{
				this->constructMousePosition(lParam);
				this->startTrackMouseEvent();
				m_mouseListener->handleMouseMovement(m_mouseEvent);
				result = 0;
			}
			else
			{
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_LBUTTONDBLCLK:
			if (m_mouseListener)
			{
				this->constructMousePosition(lParam);
				m_mouseListener->handleDoubleClick(m_mouseEvent);
				result = 0;
			}
			else
			{
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_LBUTTONDOWN:
			if (m_mouseListener)
			{
				if (!m_mouse.isCaptured())
				{
					m_mouse.capture(m_windowHandle);
				}
				this->constructMousePosition(lParam);
				this->startTrackMouseEvent();
				m_mouseListener->handleLeftButtonDown(m_mouseEvent);
				result = 0;
			}
		break;
		case WM_RBUTTONDOWN:
			if (m_mouseListener)
			{
				if (!m_mouse.isCaptured())
				{
					m_mouse.capture(m_windowHandle);
				}
				this->constructMousePosition(lParam);
				this->startTrackMouseEvent();
				m_mouseListener->handleRightButtonDown(m_mouseEvent);
				result = 0;
			}
		break;
		case WM_LBUTTONUP:
			if (m_mouseListener)
			{
				if (m_mouse.isCaptured())
				{
					m_mouse.release();
				}
				this->constructMousePosition(lParam);
				m_mouseListener->handleLeftButtonUp(m_mouseEvent);
				result = 0;
			}
		break;
		case WM_RBUTTONUP:
			if (m_mouseListener)
			{
				if (m_mouse.isCaptured())
				{
					m_mouse.release();
				}
				this->constructMousePosition(lParam);
				m_mouseListener->handleRightButtonUp(m_mouseEvent);
				result = 0;
			}
		break;
		case WM_MOUSEWHEEL:
			if (m_mouseListener)
			{
				POINT point;
				GetCursorPos(&point);

				// Get window left and right
				CRect area;
				CWindowTools::getGlobalContentAreaCoordinates(m_windowHandle, area);

				m_mousePoint.setPoint(point.x - area.getLeft(), point.y - area.getTop());
				m_mouse.setPosition(m_mousePoint);
				m_mouseEvent.setModifiers(GetAsyncKeyState(17) ? true : false, GetAsyncKeyState(16) ? true : false, GetAsyncKeyState(18) ? true : false);
				m_mouseEvent.setWheelMovementAmount(((double)GET_WHEEL_DELTA_WPARAM(wParam)) / (double) WHEEL_DELTA);

				// Pass to the listener for handling
				m_mouseListener->handleMouseScroll(m_mouseEvent);
				result = 0;
			}
		break;
		case WM_MOUSELEAVE:
			if (m_mouseListener)
			{
				// Set up the mouse and mouse event
				this->constructMousePosition(lParam);
				m_mouseListener->handleMouseLeavingArea(m_mouseEvent);
				result = 0;
			}
			else
			{
				result = DefWindowProc(windowHandle, message, wParam, lParam);
			}
		break;
		case WM_TIMER:
			//CTrace::trace("Recieved WM TIMER message");
			//this->handleTimerEvent((long)wParam);
			result = 0;
		break;
		default:
			result = DefWindowProc(windowHandle, message, wParam, lParam);
		break;
	}

	return result;
}

//	===========================================================================
void CWindow::constructMousePosition(LPARAM lParam)
{
	m_mousePoint.setPoint(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
	m_mouse.setPosition(m_mousePoint);
	m_mouseEvent.setModifiers(GetAsyncKeyState(17) ? true : false, GetAsyncKeyState(16) ? true : false, GetAsyncKeyState(18) ? true : false);
}

//	===========================================================================
void CWindow::startTrackMouseEvent()
{
	TRACKMOUSEEVENT trackMouseEvent;
	trackMouseEvent.cbSize		= sizeof(trackMouseEvent);
	trackMouseEvent.dwFlags		= TME_LEAVE;
	trackMouseEvent.hwndTrack	= m_windowHandle->m_windowHandle;
	_TrackMouseEvent(&trackMouseEvent);
}

//	===========================================================================
void CWindow::unhookKeyboardHandler()
{
	if (m_keyboardHook)
	{
		UnhookWindowsHookEx(m_keyboardHook);
		m_hookWindowHandle = NULL;
	}
}

//	===========================================================================
void CWindow::unhookWheelHandler()
{
	if (m_mouseHook)
	{
		UnhookWindowsHookEx(m_mouseHook);
		m_mouseHook = NULL;
	}
}
#else

//	===========================================================================
pascal OSStatus CWindow::WindowProc(EventHandlerCallRef handler, EventRef theEvent, void *userData)
{
	if (userData)
	{
		return ((CWindow*)userData)->handleWindowEvents(handler, theEvent);
	}
	return eventNotHandledErr;
}

//	===========================================================================
pascal OSStatus CWindow::handleWindowEvents(EventHandlerCallRef handler, EventRef theEvent)
{
	OSStatus result = eventNotHandledErr;

	UInt32 myEventClass = GetEventClass(theEvent);
	UInt32 myEventKind  = GetEventKind(theEvent);

	switch(myEventClass)
	{
		case kEventClassKeyboard:
			switch(myEventKind)
			{
				case kEventRawKeyDown:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyDown(m_keyEvent);
						result = noErr;
					}
				break;
				case kEventRawKeyUp:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyUp(m_keyEvent);
						result = noErr;
					}
				break;
				case kEventRawKeyRepeat:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyDown(m_keyEvent);
						result = noErr;
					}
				break;
			}
		break;

		case kEventClassMouse:
			switch(myEventKind)
			{
				case kEventMouseDragged:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);
						m_mouseListener->handleMouseMovement(m_mouseEvent);
						result = noErr;
					}
				break;
				case kEventMouseMoved:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);

						// Get the position
						Point point;
						GetEventParameter(theEvent, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(Point), NULL, &point);

						// Get the window handle
						WindowRef windowHandle;
						FindWindow(point, &windowHandle);

						// Get the bounds
						Rect windBounds;
						GetWindowBounds(windowHandle, kWindowContentRgn, &windBounds);

						if (PtInRect(point, &windBounds))
						{
							m_mouseListener->handleMouseMovement(m_mouseEvent);
						}
						else
						{
							m_mouseListener->handleMouseLeavingArea(m_mouseEvent);
						}
						result = noErr;
					}
				break;
				case kEventMouseWheelMoved:
					if (m_mouseListener)
					{
						SInt32 delta;
						GetEventParameter(theEvent, kEventParamMouseWheelDelta, typeLongInteger, NULL, sizeof(delta), NULL, &delta);

						this->constructMousePosition(theEvent);
						m_mouseEvent.setWheelMovementAmount((double)delta);

						m_mouseListener->handleMouseScroll(m_mouseEvent);
						result = noErr;
					}
				break;
				case kEventMouseUp:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);

						UInt16 theButton;
						GetEventParameter(theEvent, kEventParamMouseButton, typeMouseButton, NULL, sizeof(UInt16), NULL, &theButton);

						if (theButton == kEventMouseButtonPrimary || m_leftClick)
						{
							m_mouseListener->handleLeftButtonUp(m_mouseEvent);
						}
						else if (theButton == kEventMouseButtonSecondary)
						{
							m_mouseListener->handleRightButtonUp(m_mouseEvent);
						}
					}
				break;
				case kEventMouseDown:
					{
						Point point;
						GetEventParameter(theEvent, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(Point), NULL, &point);
						WindowRef windowHandle;
						const WindowPartCode partCode = FindWindow(point, &windowHandle);

						if (!IsWindowActive(windowHandle))
						{
							return eventNotHandledErr;
						}

						switch(partCode)
						{
							case inDrag:
								{
									DragWindow(windowHandle, point, NULL);

									Rect windBounds;
									GetWindowBounds(windowHandle, kWindowContentRgn, &windBounds);

									CRect positionAndSize;
									positionAndSize.setFromRect(windBounds);
									m_windowPosition.setPoint(positionAndSize.getLeft(), positionAndSize.getTop());
									result = noErr;
								}
							break;
							case inGoAway:
								if(TrackGoAway(windowHandle, point))
								{
									this->closeWindow();
									result = noErr;
								}
							break;
							case inGrow:
								{
									Rect theFinalRect;
									ResizeWindow(m_windowHandle->m_windowHandle, point, NULL, &theFinalRect);

									CDimension dimension(theFinalRect.right - theFinalRect.left, theFinalRect.bottom - theFinalRect.top);
									m_windowDimension = dimension;

									// Notify the listener
									if (m_windowChangeListener)
									{
										m_windowChangeListener->windowSized(this, dimension);
									}
									result = noErr;
								}
							break;
							case inContent:
								if (m_mouseListener)
								{
									this->constructMousePosition(theEvent);

									// Obtain the button
									UInt16 theButton;
									GetEventParameter(theEvent, kEventParamMouseButton, typeMouseButton, NULL, sizeof(UInt16), NULL, &theButton);

									// Or modifiers & command
									if (theButton == kEventMouseButtonPrimary)
									{
										// Get click count
										UInt32 clickCount;
										GetEventParameter(theEvent, kEventParamClickCount, typeUInt32, NULL, sizeof(UInt32), NULL, &clickCount);

										// Check if single or double click is required
										if (clickCount == 2)
										{
											m_mouseListener->handleDoubleClick(m_mouseEvent);
										}
										else
										{
											if (m_mouseEvent.isAltDown())
											{
												m_leftClick = false;
												m_mouseListener->handleRightButtonDown(m_mouseEvent);
											}
											else
											{
												m_leftClick = true;
												m_mouseListener->handleLeftButtonDown(m_mouseEvent);
											}
										}
									}
									else if (theButton == kEventMouseButtonSecondary)
									{
										m_leftClick = false;
										m_mouseListener->handleRightButtonDown(m_mouseEvent);
									}
									result = noErr;
								}
							break;
							case inZoomIn:
								{
									// Zoom back to the default size. We treat as a resize event
									ZoomWindow(windowHandle, partCode, true);

									Rect windBounds;
									GetWindowBounds(windowHandle, kWindowContentRgn, &windBounds);

									CRect positionAndSize;
									positionAndSize.setFromRect(windBounds);

									m_windowSize = positionAndSize;
									m_windowPosition.setPoint(positionAndSize.getLeft(), positionAndSize.getTop());
									m_windowDimension.setDimension(positionAndSize.getWidth(), positionAndSize.getHeight());

									if (m_windowChangeListener)
									{
										m_windowChangeListener->windowSized(this, m_windowDimension);
									}

									result = noErr;
								}
							break;
							case inZoomOut:
								{
									// Maximise!!
									ZoomWindow(windowHandle, partCode, true);

									Rect windBounds;
									GetWindowBounds(windowHandle, kWindowContentRgn, &windBounds);

									CRect positionAndSize;
									positionAndSize.setFromRect(windBounds);

									m_windowSize = positionAndSize;
									m_windowPosition.setPoint(positionAndSize.getLeft(), positionAndSize.getTop());
									m_windowDimension.setDimension(positionAndSize.getWidth(), positionAndSize.getHeight());

									if (m_windowChangeListener)
									{
										m_windowChangeListener->windowMaximised(this);
									}

									result = noErr;
								}
							break;
							case inProxyIcon:
								// We have no real handling for proxy icons, cos they're stupid ;)
								TrackWindowProxyDrag(windowHandle,point);
								result = noErr;
							break;
							case inCollapseBox:
								CollapseWindow(windowHandle, true);
								if (m_windowChangeListener)
								{
									m_windowChangeListener->windowMinimised(this);
								}
								result = noErr;
							break;
						}
					}
				break;
			}
		break;
		case kEventClassWindow:
			switch(myEventKind)
			{
				case kEventWindowDrawFrame:
					{
						// This means to draw the controls (collapse box etc
						return noErr;
					}
				break;
				case kEventWindowInit:
					SetThemeWindowBackground(m_windowHandle->m_windowHandle, kThemeBrushMovableModalBackground, true);
					result = noErr;
				break;
				case kEventWindowShown:
					if (m_windowChangeListener)
					{
						m_windowChangeListener->windowOpened(this);
						result = noErr;
					}
				break;
				case kEventWindowActivated:
				case kEventWindowFocusAcquired:
					this->redrawWindow();
					if (m_focusListener)
					{
						m_focusListener->gainedFocus(this);
						result = noErr;
					}
				break;
				case kEventWindowDeactivated:
				case kEventWindowFocusRelinquish:
					if (m_focusListener)
					{
						m_focusListener->lostFocus(this);
						result = noErr;
					}
				break;
				case kEventWindowClosed:
					if (m_windowChangeListener)
					{
						m_windowChangeListener->windowClosed(this);
						result = noErr;
					}
				break;
				case kEventWindowDrawContent:
					{
					
						//CTrace::trace("Window draw");
						// Get the window handle
						WindowRef windowHandle;
						GetEventParameter(theEvent, kEventParamDirectObject, typeWindowRef, NULL, sizeof(windowHandle), NULL, &windowHandle);

						// Get the current port
						CGrafPtr thePort;
						GetPort(&thePort);

						// Store the update area
						Rect updateArea;
						GetPortBounds(thePort, &updateArea);

						// Set the background colour
						RGBColor *backColour = m_attributes->getBackgroundColour().getAsRGBColour();
						RGBBackColor(backColour);
						EraseRect(&updateArea);
						FREE_POINTER(backColour);

						// Setup the qwartz context
						CGContextRef theContext;

						// Start the context drawing
						QDBeginCGContext(thePort, &theContext);

						// Clip to the area
						CGContextSaveGState(theContext);

						// Setup the graphics object
						m_graphics.initialise(m_windowHandle, theContext, m_windowDimension);

						CRect myUpdate;
						if (m_updateAreaIsValid)
						{
							myUpdate			= m_redrawUpdateArea;
							m_updateAreaIsValid = false;
						}
						else
						{
							myUpdate.setFromRect(updateArea);
						}
						m_graphics.setUpdateArea(myUpdate);
						m_graphics.resetDrawingArea();

						// Draw the images
						this->draw(m_graphics);

						// destup the graphics
						m_graphics.resetDrawingArea();
						m_graphics.uninitialise();

						// Restore the clipped rectangle
						CGContextRestoreGState(theContext);

						// Flush the content to the window
						CGContextFlush(theContext);

						// End the use of the context
						QDEndCGContext(thePort, &theContext);

						// And we are done
						result = noErr;
					}
				break;
			}
		break;
	}

	return result;
}

//	===========================================================================
pascal OSErr CWindow::handleDropMessages(WindowRef window, void *userData, DragRef dragReference)
{
	if (userData)
	{
		return ((CWindow*)userData)->handleDropEvent(window, dragReference);
	}
	return eventNotHandledErr;
}

//	===========================================================================
pascal OSErr CWindow::handleDropEvent(WindowRef window, DragRef dragReference)
{
	if (m_dropFileListener)
	{
		// Get the position of the mouse
		Point mousePosition;
		GetDragMouse(dragReference, &mousePosition, NULL);
		CPoint position(mousePosition.h, mousePosition.v);

		UInt16 numberOfItemsBeingDropped;
		CountDragItems(dragReference, &numberOfItemsBeingDropped);

		CDropEvent dropEvent(position);
		TStringCountedPointerArray *array = dropEvent.getDroppedFileNames();

		for (UInt16 i = 1; i <= numberOfItemsBeingDropped; i++)
		{
			DragItemRef referenceCount;
			GetDragItemReferenceNumber(dragReference, i, &referenceCount);

			FlavorFlags theFlavourFlags;
			if (GetFlavorFlags(dragReference, referenceCount, 'TEXT', &theFlavourFlags) == noErr)
			{
				Size dataSize;
				OSStatus error = GetFlavorDataSize(dragReference, referenceCount, 'TEXT', &dataSize);

				if (error == noErr && dataSize > 0)
				{
					char *localBuffer = new char[dataSize + 1];
					memset(localBuffer, 0, dataSize + 1 * sizeof(char));
					if (GetFlavorData(dragReference, referenceCount, 'TEXT', localBuffer, &dataSize, 0) == noErr)
					{
						array->addElement(new CString(localBuffer));
					}
					FREE_ARRAY_POINTER(localBuffer);
				}
			}
			else if (GetFlavorFlags(dragReference, referenceCount, flavorTypeHFS, &theFlavourFlags) == noErr)
			{
				Size dataSize;
				OSStatus error = GetFlavorDataSize(dragReference, referenceCount, flavorTypeHFS, &dataSize);

				if (error == noErr && dataSize > 0)
				{
					HFSFlavor theDataFlavour;
					if (GetFlavorData(dragReference, referenceCount,flavorTypeHFS, &theDataFlavour, &dataSize, 0) == noErr)
					{
						FSRef fileReference;
						FSpMakeFSRef(&theDataFlavour.fileSpec, &fileReference);
						char buffer[_MAX_PATH];
						FSRefMakePath(&fileReference, (UInt8 *)buffer, _MAX_PATH);
						array->addElement(new CString(buffer));
					}
				}
			}
		}

		// Now let the listener handle the event
		m_dropFileListener->handleFileDrop(dropEvent);

		// And we are done
		return noErr;
	}
	return eventNotHandledErr;
}

//	===========================================================================
void CWindow::constructMousePosition(EventRef theEvent)
{
	// Get the position
	Point point;
	GetEventParameter(theEvent, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(Point), NULL, &point);

	// Get the window bounds
	Rect windBounds;
	GetWindowBounds(m_windowHandle->m_windowHandle, kWindowContentRgn, &windBounds);

	// Store the position
	m_mousePoint.setPoint(point.h - windBounds.left, point.v - windBounds.top);
	m_mouse.setPosition(m_mousePoint);

	// Get the modifiers
	UInt32 modifiers;
	GetEventParameter(theEvent, kEventParamKeyModifiers, typeUInt32, NULL, sizeof(typeUInt32), NULL, &modifiers);
	m_mouseEvent.setModifiers((modifiers & controlKey) ? true : false, (modifiers & shiftKey) ? true : false, (modifiers & cmdKey) ? true : false);
}

//	===========================================================================
void CWindow::constructKeyboardEvent(EventRef theEvent)
{
	// Get the key code
	char charCode;
	GetEventParameter(theEvent, kEventParamKeyMacCharCodes, typeChar, NULL, 1, NULL, &charCode);
	m_keyEvent.setKey(charCode);

	// Get the modifiers
	UInt32 modifiers;
	GetEventParameter(theEvent, kEventParamKeyModifiers, typeUInt32, NULL, sizeof(typeUInt32), NULL, &modifiers);
	m_keyEvent.setModifiers((modifiers & controlKey) ? true : false, (modifiers & shiftKey) ? true : false, (modifiers & cmdKey) ? true : false);
}
#endif

//	===========================================================================
void CWindow::referenced()
{
	m_referenceCount++; 
}

//	===========================================================================
void CWindow::dereference()
{
	m_referenceCount--;
	if (m_referenceCount <= 0)
	{
		delete this;
	}
}

//	===========================================================================
long CWindow::getReferenceCount() const
{
	return m_referenceCount; 
}

//	===========================================================================
void CWindow::getObjectDescription(char *string, const long size) const
{
	// Dont do anything for the moment
}